package control;
/**
 * @author Concetta Balestra, Hani Qusa
 * @version 1.2 09/03/20
 * @since JDK 1.6
 */
import java.util.*;

import abstraction.AvailableServices;
import abstraction.OnlyReadIterator;
import abstraction.Service;
import abstraction.State;


/**
 * This class permits to create the Cartesian product of all community states and all final states of a set of available services
 *
 */
public class Community {
	
	private AvailableServices availableService;
	private Set<CState> CStates; //containing all community states	
	private Set<CState> CFStates;	//containing all community states	
	
	public Community(AvailableServices as){	
		availableService = as;
		CStates = new HashSet<CState>();
		CFStates = new HashSet<CState>();	
	}
	
	/**
	 * creates all states of the community as a Cartesian product of states of available services 
	 * @param s - first service of the community
	 * @param loop - the position in a community state where you want to place the states of a service
	 */
	private void createCommunityStates (Service s, int loop, CState CSRecord) {
		Iterator<State> state = s.getStates();
		while (state.hasNext()) {
			State st = (State)state.next();
			if (loop < availableService.sizeCommunity()-1) {
				//CSRecord.add(loop, st);
				CSRecord = CSRecord.buildNewState(loop, st);
				
				createCommunityStates ((Service)availableService.get(loop+1), loop+1, CSRecord);
			}
			else {
				//CSRecord.add(loop, st);
				//CState r = (CState)CSRecord.clone();
				//CStates.add(r);
				
				CSRecord = CSRecord.buildNewState(loop, st);
				CStates.add(CSRecord);
			}
		}
	}
	
	/**
	 * returns all community states created with a Cartesian product of all states of available services
	 * @return a set of community states
	 */
	public Iterator<CState> getCommunityStates () {
		if (CStates.isEmpty()) {
			//CState CSRecord = new CState (this.availableService.sizeCommunity());
			CState CSRecord = new CState (new State[this.availableService.sizeCommunity()]);
			this.createCommunityStates(this.availableService.get(0), 0, CSRecord);
		}
		
		return new OnlyReadIterator<CState>(CStates.iterator());
	}

	/**
	 * creates all final states of the community as a Cartesian product of final states of available services 
	 * @param s - first service of the community
	 * @param loop - the position in a community state where you want to place the states of a service
	 */
	private void createFinalCommunityState (Service s, int loop, CState CSRecord) {
		Iterator<State> state = s.getFinalStates();
		while (state.hasNext()) {
			State st = (State)state.next();
			if (loop < availableService.sizeCommunity()-1) {
				//CSRecord.add(loop, st);
				CSRecord = CSRecord.buildNewState(loop, st);
				createFinalCommunityState ((Service)availableService.get(loop+1), loop+1, CSRecord);
			}
			else {
				//CSRecord.add(loop, st);
				//CState r = (CState)CSRecord.clone();
				//CFStates.add(r);
				
				CSRecord = CSRecord.buildNewState(loop, st);
				CFStates.add(CSRecord);
			}
		}
	}
	
	/**
	 * returns community states which are final states as a Cartesian product of all final states of every available services
	 * @return a set of final community states
	 */
	public Iterator<CState> getFinalCommunityStates () {
		if (this.CFStates.isEmpty()) {
			//CState CSRecord = new CState (this.availableService.sizeCommunity());
			CState CSRecord = new CState (new State[this.availableService.sizeCommunity()]);
			this.createFinalCommunityState(this.availableService.get(0), 0, CSRecord);
		}
		return new OnlyReadIterator<CState>(this.CFStates.iterator());
	}

	
	/**
	 * gets the service in the specified position
	 * @param index - the position
	 * @return a service
	 */
	public Service getService (int index) {
		return this.availableService.get(index);
	}
	
	
	/**
	 * gets a community state that is final
	 * @return CState
	 */
	public CState getInitialStates () {
		State[] initial  = new State [this.availableService.sizeCommunity()];
		for (int i = 0; i < this.availableService.sizeCommunity(); i++) {
			State s = this.availableService.get(i).getInitialState();
			initial[i] =s;
		}
		return new CState(initial);
	}
	
	/**
	 * gets the position of a service in the set of available services
	 * @param service - the service we want to find
	 * @return an integer
	 */
	public int getServicePosition (Service service) {
		return this.availableService.getPosition(service);
	}
	
	/**
	 * checks if a state of the community is final
	 * @param cs - the community's state to check
	 * @return true if cs is final
	 */
	public boolean isFinalCommunityState(CState cs){
		return this.CFStates.contains(cs);
	}
	
	public String toString () {
		StringBuilder s = new StringBuilder();
		Iterator<CState> it = this.getCommunityStates();
		while (it.hasNext()) {
			CState c = it.next();
			s = s.append("(" + c + ")");
		}
		return s.toString();
	}
	
	public int size()
	{
		return this.availableService.sizeCommunity();
	}
}

